<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('WC_Email_Customer_RFQ_Sent')) :

    /**
     * Customer Processing RFQ Email
     *
     * An email sent to the customer when a new RFQ is received.
     * @extends     WC_Email
     */
    #[\AllowDynamicProperties]
    class WC_Email_Customer_RFQ_Sent extends WC_Email
    {
        public $content_intro;
        public $_templates;

        /**
         * Constructor
         */
        function __construct()
        {

            $this->id = 'customer_rfq_sent';
            $this->title = __('RFQ-ToolKit Quote Sent', 'rfqtk');
            $this->description = __('This is the quote sent to customers containing their quote detait. 
            The content of this email can be enhanced by using the proposal section in the order.', 'rfqtk');

           // $this->heading = __('Your Quote for Request #({order_number})', 'rfqtk');

            $this->content_intro = $this->format_string($this->get_option( 'content_intro' ));

            $this->template_html = 'emails/customer-rfq-sent.php';
            $this->template_plain = 'emails/plain/customer-rfq-sent.php';
            $this->_templates = array($this->template_html, $this->template_plain);

            $this->placeholders   = array(
                '{order_date}'   => '',
                '{order_number}' => '',
            );

            // Triggers for this email

            $this->customer_email = true;
            $this->recipient = $this->get_option( 'recipient', '' );

            add_filter('woocommerce_template_directory', array($this, 'gpls_rfq_woocommerce_locate_template_dir'), 10, 2);

            if(!has_action('woocommerce_order_status_gplsquote-sent_notification', array($this, 'trigger'), 100, 1)) {
                add_action('woocommerce_order_status_gplsquote-sent_notification', array($this, 'trigger'), 100, 1);
            }

            // Call parent constructor
            parent::__construct();


        }

        public function rfqtk_email_proposal_function($order, $sent_to_admin, $plain_text)
        {

             wc_get_template('woo-rfq/rfqtk-proposal.php',
                array('order' => $order,'sent_to_admin'=>$sent_to_admin,'plain_text'=>$plain_text)
                 ,'',gpls_woo_rfq_plus_DIR . 'woocommerce/');

        }
        public function get_default_subject()
        {
            return __('Your Quote Request #({order_number}) from [{site_title}] on {order_date}', 'rfqtk');
        }

        public function gpls_rfq_woocommerce_locate_template_dir($dir, $template)
        {
            return $dir;
        }

        public function get_default_heading() {
            return __('Your Quote for Request #({order_number})', 'rfqtk');
        }

        /**
         * Trigger.
         */
        function trigger($order_id)
        {

            if(defined("WC_Email_Customer_RFQ_Sent".$order_id))return;
            define("WC_Email_Customer_RFQ_Sent".$order_id,true);

            $this->setup_locale();

            if(!has_action('rfqtk_email_proposal',array($this, 'rfqtk_email_proposal_function'))) {
                add_action('rfqtk_email_proposal', array($this, 'rfqtk_email_proposal_function'), 100, 3);
            }
            if ($order_id) {
                $order = WC_Order_Factory::get_order($order_id);
                $order->update_meta_data('_gpls_woo_is_quote','yes');
                $order->save();
            }

            if ($order_id) {
                $this->object = wc_get_order($order_id);
                $this->recipient = $this->get_option( 'recipient').','.$this->object->get_billing_email();

                $this->placeholders['{order_date}']   = wc_format_datetime( $this->object->get_date_created() );
                $this->placeholders['{order_number}'] = $this->object->get_order_number();

            }



            if ( $this->is_enabled() && $this->get_recipient() )
            {

                try {
                    $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
                }catch(Exception $ex){np_write_log($ex,__FILE__,__LINE__);}
            }

            $this->restore_locale();



        }


        function replace_placeholders( $order_id ) {

            if (defined("WC_Email_Customer_RFQ_Sent_EMAIL" . $order_id)) return;

            define("WC_Email_Customer_RFQ_Sent_EMAIL" . $order_id, true);

            require_once(WC()->plugin_path() . '/includes/emails/class-wc-email.php');
            require_once(WC()->plugin_path() . '/includes/class-wc-emails.php');

            $WC_Emails = WC_Emails::instance();

            $this->setup_locale();

            if ( $order_id ) {
                $this->object       = wc_get_order( $order_id );


                $this->placeholders['{order_date}']   = wc_format_datetime( $this->object->get_date_created() );
                $this->placeholders['{order_number}'] = $this->object->get_order_number();
                $this->placeholders['{order_billing_full_name}'] = $this->object->get_formatted_billing_full_name();

            }

            $this->restore_locale();

        }




        /**
         * get_content_html function.
         *
         * @access public
         * @return string
         */
        function get_content_html()
        {


            return wc_get_template_html($this->template_html, array(
                'order' => $this->object,
                'email_heading' => $this->get_heading(),
                'additional_content' => is_callable(array($this, 'get_additional_content'))?$this->get_additional_content():"",
                'content_intro' => $this->content_intro,
                'sent_to_admin' => false,
                'plain_text' => false,
                'email'			=> $this,
            ),'',gpls_woo_rfq_plus_DIR . 'woocommerce/');

        }

        /**
         * get_content_plain function.
         *
         * @access public
         * @return string
         */
        function get_content_plain()
        {
            return wc_get_template_html($this->template_plain, array(
                'order' => $this->object,
                'email_heading' => $this->get_heading(),
                'additional_content' => is_callable(array($this, 'get_additional_content'))?$this->get_additional_content():"",
                'content_intro' => $this->content_intro,
                'sent_to_admin' => false,
                'plain_text' => true,
                'email'			=> $this,

            ),'',gpls_woo_rfq_plus_DIR . 'woocommerce/');

        }

        /**
         * Initialise settings form fields
         */
        public function init_form_fields()
        {
            $placeholder_text  = sprintf( __( 'Available placeholders: %s', 'rfqtk' ), '<code>' . implode( '</code>, <code>', array_keys( $this->placeholders ) ) . '</code>' );

            $this->form_fields = array(
                'enabled' => array(
                    'title' => __('Enable/Disable', 'rfqtk'),
                    'type' => 'checkbox',
                    'label' => __('Enable this email notification', 'rfqtk'),
                    'default' => 'yes'
                ),
                'recipient'  => array(
                    'title'       => __( 'Additional Recipient(s)', 'rfqtk' ),
                    'type'        => 'text',
                    /* translators: %s: WP admin email */
                    'description' => sprintf( __( 'Enter recipients ( in addition to the customer) comma separated for this email.', 'rfqtk' ), '<code>' . esc_attr( get_option( 'admin_email' ) ) . '</code>' ),
                    'placeholder' => '',
                    'default'     => '',
                    'desc_tip'    => true,
                ),

                'subject' => array(
                    'title'         => __( 'Subject', 'rfqtk' ),
                    'description' => $placeholder_text,
                    'placeholder' => __($this->get_default_subject(), 'rfqtk' ),
                    'css'         => 'width:600px',
                    'default'       => ''
                ),
                'heading'            => array(
                    'title'       => __( 'Email heading', 'rfqtk' ),
                    'type'        => 'text',
                    'desc_tip'    => true,
                    'description' => $placeholder_text,
                    'placeholder' => $this->get_default_heading(),
                    'default'     => '',
                ),

                'content_intro' => array(
                    'title' => __('Content Intro', 'rfqtk'),
                    'type' => 'textarea',
                    'description' => sprintf(__('This controls the first paragrah contained within the email notification.', 'rfqtk'), $this->heading),
                    'placeholder' => __( 'N/A', 'rfqtk' ),
                    'css'         => 'width:400px; height: 75px;',
                    'default' => '',
                    'desc_tip'    => true,

                ),
                'additional_content' => array(
                    'title'       => __( 'Additional content', 'rfqtk' ),
                    'description' => __( 'Text to appear below the main email content.', 'rfqtk' ),
                    'css'         => 'width:400px; height: 75px;',
                    'placeholder' => __( 'N/A', 'rfqtk' ),
                    'type'        => 'textarea',
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'email_type' => array(
                    'title' => __('Email type', 'rfqtk'),
                    'type' => 'select',
                    'description' => __('Choose which format of email to send.', 'rfqtk'),
                    'default' => 'html',
                    'class' => 'email_type wc-enhanced-select',
                    'options' => $this->get_email_type_options()
                )
            );
        }


    }

endif;

return new WC_Email_Customer_RFQ_Sent();
